<?php
/* --------------------------------------------------------------
   FetchAllTrackingCodesAction.php 2020-04-09
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Api\TrackingCode\Actions;

use Gambio\Admin\TrackingCode\Interfaces\TrackingCodeService;
use Gambio\Api\Application\Responses\CreateApiMetaDataTrait;
use Gambio\Api\Application\Responses\ResponseDataTrimmerTrait;
use Gambio\Api\TrackingCode\TrackingCodeApiRequestParser;
use Slim\Http\Response;
use Slim\Http\ServerRequest as Request;

/**
 * Class FetchAllTrackingCodesAction
 *
 * @package Gambio\Api\TrackingCode\Actions
 */
class FetchAllTrackingCodesAction
{
    use CreateApiMetaDataTrait;
    use ResponseDataTrimmerTrait;
    
    /**
     * @var TrackingCodeApiRequestParser
     */
    private $requestParser;
    
    /**
     * @var TrackingCodeService
     */
    private $service;
    
    
    /**
     * FetchAllTrackingCodesAction constructor.
     *
     * @param TrackingCodeApiRequestParser $requestParser
     * @param TrackingCodeService          $service
     */
    public function __construct(
        TrackingCodeApiRequestParser $requestParser,
        TrackingCodeService $service
    ) {
        $this->requestParser = $requestParser;
        $this->service       = $service;
    }
    
    
    /**
     * @param Request  $request
     * @param Response $response
     * @param array    $args
     *
     * @return Response
     */
    public function __invoke(Request $request, Response $response, array $args): Response
    {
        $criteria = $this->requestParser->getCriteriaFromRequest($request);
        $fields   = $this->requestParser->getFieldsFromRequest($request);
        
        $trackingCodes = $this->service->getAllTrackingCodes($criteria,
                                                             $this->requestParser->getPaginationFromRequest($request));
        $totalItems    = $this->service->getTotalCountOfTrackingCodes($criteria);
        $metaData      = $this->createApiCollectionMetaData($this->requestParser->getPageFromRequest($request),
                                                            $this->requestParser->getPerPageFromRequest($request),
                                                            $totalItems,
                                                            $this->requestParser->getResourceUrlFromRequest($request),
                                                            $request->getQueryParams());
        
        $responseData = $trackingCodes;
        if (count($fields) > 0) {
            $collectionData = json_decode(json_encode($trackingCodes), true);
            $responseData   = $this->trimCollectionData($collectionData, $fields);
        }
        
        return $response->withJson([
                                       'data'  => $responseData,
                                       '_meta' => $metaData,
                                   ]);
    }
}